<?php
use yii\helpers\Html;
use yii\helpers\Url;
use yii\widgets\ActiveForm;

$this->title = 'Каталог туров';
?>

<?php $this->registerCssFile('https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css') ?>
<?php $this->registerCssFile('https://fonts.googleapis.com/css2?family=Manrope:wght@300;400;500;600;700&display=swap') ?>

<style>
:root {
    --bg: #fcfdfd;
    --surface: #ffffff;
    --text: #2a2f3a;
    --text-light: #6c757d;
    --primary: #4ecdc4;
    --primary-light: #b0e5e1;
    --accent: #ffd166;
    --shadow-sm: 0 4px 12px rgba(0,0,0,0.03);
    --shadow-md: 0 8px 24px rgba(0,0,0,0.05);
    --radius: 20px;
    --transition: all 0.3s cubic-bezier(0.25, 0.8, 0.25, 1);
}

* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: 'Manrope', sans-serif;
    background-color: var(--bg);
    color: var(--text);
}

.catalog-page {
    min-height: calc(100vh - 160px);
    padding: 60px 0;
}

.catalog-container {
    max-width: 1400px;
    margin: 0 auto;
    padding: 0 30px;
    display: grid;
    grid-template-columns: 280px 1fr;
    gap: 40px;
}

/* Header */
.catalog-header {
    grid-column: 1 / -1;
    text-align: center;
    margin-bottom: 50px;
}

.catalog-title {
    font-size: 2.8rem;
    font-weight: 700;
    margin-bottom: 16px;
    background: linear-gradient(135deg, var(--text), var(--primary));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

.catalog-subtitle {
    color: var(--text-light);
    font-size: 1.1rem;
    max-width: 600px;
    margin: 0 auto;
}

/* Filters */
.filters-panel {
    background: var(--surface);
    border-radius: var(--radius);
    padding: 28px;
    box-shadow: var(--shadow-sm);
    position: sticky;
    top: 100px;
}

.filters-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 24px;
}

.filters-title {
    font-size: 1.3rem;
    font-weight: 600;
    color: var(--text);
}

.clear-filters {
    background: transparent;
    border: none;
    color: var(--primary);
    font-size: 0.9rem;
    cursor: pointer;
    padding: 6px 14px;
    border-radius: 12px;
    transition: var(--transition);
}

.clear-filters:hover {
    background: var(--primary-light);
}

.filter-group {
    margin-bottom: 30px;
    padding: 20px;
    background: rgba(248, 249, 250, 0.6);
    border-radius: 16px;
    border: 1px solid rgba(78, 205, 196, 0.1);
}

.filter-group-title {
    font-size: 1.05rem;
    font-weight: 600;
    margin-bottom: 16px;
    display: flex;
    align-items: center;
    gap: 10px;
    color: var(--text);
}

.custom-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid #e0e6eb;
    border-radius: 12px;
    font-size: 0.95rem;
    background: white;
    transition: var(--transition);
}

.custom-input:focus {
    outline: none;
    border-color: var(--primary);
    box-shadow: 0 0 0 3px rgba(78, 205, 196, 0.15);
}

.custom-input-group {
    display: flex;
    gap: 12px;
    margin-bottom: 16px;
}

.price-range-container {
    margin-top: 16px;
}

.price-range {
    height: 6px;
    background: #e0e6eb;
    border-radius: 3px;
    position: relative;
    margin: 16px 0;
}

.price-range-fill {
    position: absolute;
    height: 100%;
    background: var(--primary);
    border-radius: 3px;
}

.price-range-thumb {
    width: 18px;
    height: 18px;
    background: white;
    border: 2px solid var(--primary);
    border-radius: 50%;
    position: absolute;
    top: 50%;
    transform: translateY(-50%);
    cursor: pointer;
    box-shadow: 0 2px 6px rgba(0,0,0,0.1);
}

.price-labels {
    display: flex;
    justify-content: space-between;
    color: var(--text-light);
    font-size: 0.85rem;
}

.filter-button {
    width: 100%;
    padding: 14px;
    background: var(--primary);
    color: white;
    border: none;
    border-radius: 14px;
    font-weight: 600;
    font-size: 1rem;
    cursor: pointer;
    transition: var(--transition);
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 10px;
}

.filter-button:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(78, 205, 196, 0.25);
}

/* Sorting */
.sorting-panel {
    background: var(--surface);
    border-radius: 16px;
    padding: 20px 28px;
    box-shadow: var(--shadow-sm);
    display: flex;
    justify-content: space-between;
    align-items: center;
    grid-column: 2;
    margin-bottom: 30px;
}

.sorting-label {
    font-weight: 600;
    color: var(--text);
    display: flex;
    align-items: center;
    gap: 8px;
}

.sorting-options {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.sort-option {
    padding: 8px 20px;
    background: white;
    border: 1px solid #e0e6eb;
    border-radius: 50px;
    color: var(--text-light);
    font-size: 0.9rem;
    font-weight: 500;
    text-decoration: none;
    transition: var(--transition);
    white-space: nowrap;
}

.sort-option:hover {
    border-color: var(--primary);
    color: var(--primary);
}

.sort-option.active {
    background: var(--primary);
    color: white;
    border-color: var(--primary);
}

/* Tours Grid */
.tours-grid {
    grid-column: 2;
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
    gap: 32px;
}

.tour-card {
    background: var(--surface);
    border-radius: var(--radius);
    overflow: hidden;
    box-shadow: var(--shadow-sm);
    transition: var(--transition);
    height: 100%;
    display: flex;
    flex-direction: column;
}

.tour-card:hover {
    transform: translateY(-8px);
    box-shadow: var(--shadow-md);
}

.tour-img {
    height: 200px;
    background-size: cover;
    background-position: center;
    position: relative;
}

.tour-badge {
    position: absolute;
    top: 16px;
    right: 16px;
    background: var(--accent);
    color: var(--text);
    padding: 6px 16px;
    border-radius: 50px;
    font-weight: 700;
    font-size: 0.9rem;
}

.tour-content {
    padding: 24px;
    flex: 1;
    display: flex;
    flex-direction: column;
}

.tour-title {
    font-size: 1.25rem;
    font-weight: 600;
    margin-bottom: 12px;
    color: var(--text);
}

.tour-desc {
    color: var(--text-light);
    font-size: 0.95rem;
    line-height: 1.6;
    margin-bottom: 20px;
    flex: 1;
}

.tour-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    border-top: 1px solid #f0f2f5;
    padding-top: 20px;
}

.tour-price {
    font-size: 1.4rem;
    font-weight: 700;
    color: var(--primary);
}

.tour-date {
    color: var(--text-light);
    font-size: 0.9rem;
    display: flex;
    align-items: center;
    gap: 6px;
}

.tour-btn {
    display: block;
    width: 100%;
    padding: 12px;
    text-align: center;
    background: transparent;
    border: 2px solid var(--primary);
    color: var(--primary);
    border-radius: 14px;
    font-weight: 600;
    text-decoration: none;
    transition: var(--transition);
    margin-top: 16px;
}

.tour-btn:hover {
    background: var(--primary);
    color: white;
}

/* No tours */
.no-tours {
    grid-column: 2;
    background: var(--surface);
    border-radius: var(--radius);
    padding: 60px 40px;
    text-align: center;
    box-shadow: var(--shadow-sm);
}

.no-tours-icon {
    width: 80px;
    height: 80px;
    background: var(--primary-light);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 24px;
    color: var(--primary);
    font-size: 2rem;
}

.no-tours-title {
    font-size: 1.8rem;
    font-weight: 600;
    margin-bottom: 16px;
}

.no-tours-text {
    color: var(--text-light);
    max-width: 500px;
    margin: 0 auto 30px;
}

/* Responsive */
@media (max-width: 1100px) {
    .catalog-container {
        grid-template-columns: 1fr;
        gap: 30px;
    }
    .filters-panel { position: static; }
    .sorting-panel, .tours-grid, .no-tours { grid-column: 1; }
}

@media (max-width: 768px) {
    .catalog-container { padding: 0 20px; }
    .catalog-title { font-size: 2.2rem; }
    .tours-grid { grid-template-columns: 1fr; }
    .sorting-panel { flex-direction: column; align-items: flex-start; gap: 16px; }
    .sorting-options { width: 100%; overflow-x: auto; padding-bottom: 10px; }
}

@media (max-width: 576px) {
    .catalog-title { font-size: 1.8rem; }
    .filter-button { justify-content: center; }
    .tour-meta { flex-direction: column; gap: 12px; align-items: flex-start; }
}
</style>

<div class="catalog-page">
    <div class="catalog-container">
        <div class="catalog-header">
            <h1 class="catalog-title">Каталог туров</h1>
            <p class="catalog-subtitle">Найдите идеальное путешествие среди наших уникальных маршрутов и направлений</p>
        </div>

        <!-- Filters -->
        <div class="filters-panel">
            <div class="filters-header">
                <h2 class="filters-title">Фильтры</h2>
                <button type="button" class="clear-filters" id="clearFilters">
                    <i class="fas fa-times"></i> Сбросить
                </button>
            </div>

            <?php $form = ActiveForm::begin([
                'id' => 'filter-form',
                'method' => 'get',
                'action' => ['index'],
                'options' => ['class' => 'custom-form']
            ]); ?>

            <div class="filter-group">
                <h3 class="filter-group-title"><i class="fas fa-tags"></i> Цена, ₽</h3>
                <div class="custom-input-group">
                    <input type="number" name="min_price" value="<?= Yii::$app->request->get('min_price') ?>"
                           class="custom-input" placeholder="От" id="minPrice">
                    <input type="number" name="max_price" value="<?= Yii::$app->request->get('max_price') ?>"
                           class="custom-input" placeholder="До" id="maxPrice">
                </div>

                <div class="price-range-container">
                    <div class="price-range" id="priceRange">
                        <div class="price-range-fill" id="priceRangeFill"></div>
                        <div class="price-range-thumb" id="minPriceThumb"></div>
                        <div class="price-range-thumb" id="maxPriceThumb"></div>
                    </div>
                    <div class="price-labels">
                        <span id="minPriceLabel">0 ₽</span>
                        <span id="maxPriceLabel">100 000 ₽</span>
                    </div>
                </div>
            </div>

            <div class="filter-group">
                <h3 class="filter-group-title"><i class="far fa-calendar-alt"></i> Даты тура</h3>
                <div style="margin-bottom: 16px;">
                    <label style="display: block; margin-bottom: 8px; font-size: 0.9rem; color: var(--text-light);">Начало от</label>
                    <input type="date" name="date_from" value="<?= Yii::$app->request->get('date_from') ?>"
                           class="custom-input" id="dateFrom">
                </div>
                <div>
                    <label style="display: block; margin-bottom: 8px; font-size: 0.9rem; color: var(--text-light);">Окончание до</label>
                    <input type="date" name="date_to" value="<?= Yii::$app->request->get('date_to') ?>"
                           class="custom-input" id="dateTo">
                </div>
            </div>

            <button type="submit" class="filter-button">
                <i class="fas fa-search"></i> Применить
            </button>

            <?php ActiveForm::end(); ?>
        </div>

        <!-- Sorting -->
        <div class="sorting-panel">
            <div class="sorting-label">
                <i class="fas fa-sort"></i> Сортировать по:
            </div>
            <div class="sorting-options">
                <?php
                $currentSort = Yii::$app->request->get('sort', 'date_start');
                $currentOrder = Yii::$app->request->get('order', 'asc');
                $queryParams = [
                    'min_price' => Yii::$app->request->get('min_price'),
                    'max_price' => Yii::$app->request->get('max_price'),
                    'date_from' => Yii::$app->request->get('date_from'),
                    'date_to' => Yii::$app->request->get('date_to')
                ];

                $options = [
                    ['sort' => 'date_start', 'order' => 'asc', 'label' => 'Дате ↑', 'icon' => 'calendar-alt'],
                    ['sort' => 'date_start', 'order' => 'desc', 'label' => 'Дате ↓', 'icon' => 'calendar-alt'],
                    ['sort' => 'price', 'order' => 'asc', 'label' => 'Цене ↑', 'icon' => 'ruble-sign'],
                    ['sort' => 'price', 'order' => 'desc', 'label' => 'Цене ↓', 'icon' => 'ruble-sign'],
                ];

                foreach ($options as $opt):
                    $active = ($currentSort === $opt['sort'] && $currentOrder === $opt['order']);
                    $url = Url::to(array_merge(['index'], $queryParams, [
                        'sort' => $opt['sort'],
                        'order' => $opt['order']
                    ]));
                    ?>
                    <a href="<?= $url ?>" class="sort-option <?= $active ? 'active' : '' ?>">
                        <i class="fas fa-<?= $opt['icon'] ?>"></i> <?= $opt['label'] ?>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>

        <!-- Tours or Empty -->
        <?php if (empty($tours)): ?>
            <div class="no-tours">
                <div class="no-tours-icon"><i class="fas fa-compass"></i></div>
                <h3 class="no-tours-title">Туры не найдены</h3>
                <p class="no-tours-text">По вашему запросу ничего не найдено. Попробуйте изменить параметры фильтрации.</p>
                <button class="filter-button" style="max-width: 220px; margin: 0 auto;" onclick="document.getElementById('clearFilters').click()">
                    <i class="fas fa-redo"></i> Сбросить фильтры
                </button>
            </div>
        <?php else: ?>
            <div class="tours-grid">
                <?php foreach ($tours as $tour): ?>
                    <div class="tour-card">
                        <div class="tour-img" style="background-image: url('<?= $tour->image ? Yii::getAlias('@web/images/') . $tour->image : 'https://images.unsplash.com/photo-1506929562872-bb421503ef21?w=800&fit=crop' ?>')">
                            <div class="tour-badge"><?= number_format($tour->price, 0, '', ' ') ?> ₽</div>
                        </div>
                        <div class="tour-content">
                            <h3 class="tour-title"><?= Html::encode($tour->title) ?></h3>
                            <p class="tour-desc">
                                <?= mb_strlen($tour->description) > 120 ? mb_substr($tour->description, 0, 120) . '...' : $tour->description ?>
                            </p>
                            <div class="tour-meta">
                                <div class="tour-price"><?= number_format($tour->price, 0, '', ' ') ?> ₽</div>
                                <div class="tour-date">
                                    <i class="far fa-calendar-alt"></i>
                                    <?= date('d.m.Y', strtotime($tour->date_start)) ?> – <?= date('d.m.Y', strtotime($tour->date_end)) ?>
                                </div>
                            </div>
                            <a href="<?= Url::to(['catalog/view', 'id' => $tour->id]) ?>" class="tour-btn">
                                Подробнее
                            </a>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Ценовой диапазон
    const minPriceInput = document.getElementById('minPrice');
    const maxPriceInput = document.getElementById('maxPrice');
    const minPriceThumb = document.getElementById('minPriceThumb');
    const maxPriceThumb = document.getElementById('maxPriceThumb');
    const priceRange = document.getElementById('priceRange');
    const priceRangeFill = document.getElementById('priceRangeFill');
    const minPriceLabel = document.getElementById('minPriceLabel');
    const maxPriceLabel = document.getElementById('maxPriceLabel');

    const minPrice = 0;
    const maxPrice = 100000;
    const priceStep = 1000;

    let currentMin = parseInt(minPriceInput.value) || minPrice;
    let currentMax = parseInt(maxPriceInput.value) || maxPrice;

    function formatPrice(p) {
        return p.toString().replace(/\B(?=(\d{3})+(?!\d))/g, " ") + ' ₽';
    }

    function updatePriceRange() {
        const rangeWidth = priceRange.offsetWidth;
        const minPerc = ((currentMin - minPrice) / (maxPrice - minPrice)) * 100;
        const maxPerc = ((currentMax - minPrice) / (maxPrice - minPrice)) * 100;

        minPriceThumb.style.left = `calc(${minPerc}% - 9px)`;
        maxPriceThumb.style.left = `calc(${maxPerc}% - 9px)`;
        priceRangeFill.style.left = `${minPerc}%`;
        priceRangeFill.style.width = `${maxPerc - minPerc}%`;
        minPriceLabel.textContent = formatPrice(currentMin);
        maxPriceLabel.textContent = formatPrice(currentMax);
        minPriceInput.value = currentMin;
        maxPriceInput.value = currentMax;
    }

    function getPriceFromPosition(x) {
        const rect = priceRange.getBoundingClientRect();
        let price = Math.round(((x - rect.left) / rect.width) * (maxPrice - minPrice) / priceStep) * priceStep;
        return Math.max(minPrice, Math.min(maxPrice, price));
    }

    let activeThumb = null;
    function startDrag(e) { activeThumb = e.target; }
    function drag(e) {
        if (!activeThumb) return;
        const price = getPriceFromPosition(e.clientX);
        if (activeThumb === minPriceThumb) {
            currentMin = Math.min(price, currentMax - priceStep);
        } else {
            currentMax = Math.max(price, currentMin + priceStep);
        }
        updatePriceRange();
    }
    function stopDrag() { activeThumb = null; }

    minPriceThumb.addEventListener('mousedown', startDrag);
    maxPriceThumb.addEventListener('mousedown', startDrag);
    document.addEventListener('mousemove', drag);
    document.addEventListener('mouseup', stopDrag);

    minPriceInput.addEventListener('input', () => {
        let v = parseInt(minPriceInput.value) || minPrice;
        currentMin = Math.max(minPrice, Math.min(v, currentMax - priceStep));
        updatePriceRange();
    });
    maxPriceInput.addEventListener('input', () => {
        let v = parseInt(maxPriceInput.value) || maxPrice;
        currentMax = Math.min(maxPrice, Math.max(v, currentMin + priceStep));
        updatePriceRange();
    });

    priceRange.addEventListener('click', (e) => {
        const price = getPriceFromPosition(e.clientX);
        if (Math.abs(price - currentMin) < Math.abs(price - currentMax)) {
            currentMin = Math.min(price, currentMax - priceStep);
        } else {
            currentMax = Math.max(price, currentMin + priceStep);
        }
        updatePriceRange();
    });

    // Сброс фильтров
    document.getElementById('clearFilters').addEventListener('click', () => {
        minPriceInput.value = '';
        maxPriceInput.value = '';
        document.getElementById('dateFrom').value = '';
        document.getElementById('dateTo').value = '';
        currentMin = minPrice;
        currentMax = maxPrice;
        updatePriceRange();
    });

    // Даты
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('dateFrom').min = today;
    document.getElementById('dateTo').min = today;
});
</script>