<?php

/** @var yii\web\View $this */
/** @var yii\bootstrap5\ActiveForm $form */

/** @var app\models\RegisterForm $model */

use yii\bootstrap5\ActiveForm;
use yii\bootstrap5\Html;
use yii\widgets\MaskedInput;

$this->title = 'Регистрация';
$this->params['breadcrumbs'][] = $this->title;
?>
<style>
    :root {
        --dark-green: #1a535c;
        --mint: #4ecdc4;
        --cream: #f7fff7;
        --coral: #ff6b6b;
        --yellow: #ffe66d;
        --dark: #292f36;
        --gray: #6c757d;
        --light-gray: #e8f4f8;
    }

    .register-page {
        min-height: calc(100vh - 200px);
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 40px 20px;
        position: relative;
        overflow: hidden;
        background: linear-gradient(135deg, #f7fff7 0%, #e8f4f8 100%);
    }

    .register-container {
        display: flex;
        max-width: 1400px;
        width: 100%;
        min-height: 850px;
        background: white;
        border-radius: 30px;
        overflow: hidden;
        box-shadow: 0 30px 60px rgba(0, 0, 0, 0.15);
        position: relative;
    }

    /* Левая панель с шагами */
    .register-steps {
        flex: 0 0 400px;
        background: linear-gradient(135deg, var(--dark-green) 0%, #2a7a87 100%);
        padding: 60px 40px;
        color: white;
        position: relative;
        overflow: hidden;
        display: flex;
        flex-direction: column;
    }

    .register-steps::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: url('https://images.unsplash.com/photo-1519681393784-d120267933ba?w=800&fit=crop&auto=format') center/cover;
        opacity: 0.1;
    }

    .steps-header {
        position: relative;
        z-index: 1;
        margin-bottom: 60px;
    }

    .steps-title {
        font-size: 2.5rem;
        font-weight: 700;
        margin-bottom: 15px;
        line-height: 1.2;
    }

    .steps-subtitle {
        font-size: 1.1rem;
        opacity: 0.9;
        line-height: 1.6;
    }

    /* Список шагов */
    .steps-list {
        position: relative;
        z-index: 1;
        flex: 1;
    }

    .step-item {
        display: flex;
        align-items: flex-start;
        margin-bottom: 40px;
        position: relative;
        opacity: 0.7;
        transition: all 0.3s ease;
    }

    .step-item.active {
        opacity: 1;
    }

    .step-item.completed {
        opacity: 1;
    }

    .step-number {
        width: 50px;
        height: 50px;
        background: rgba(255, 255, 255, 0.15);
        border-radius: 15px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.3rem;
        font-weight: 600;
        margin-right: 20px;
        flex-shrink: 0;
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .step-item.active .step-number {
        background: linear-gradient(45deg, var(--yellow), var(--mint));
        transform: scale(1.1);
        box-shadow: 0 5px 15px rgba(255, 230, 109, 0.3);
    }

    .step-item.completed .step-number {
        background: linear-gradient(45deg, var(--mint), var(--coral));
    }

    .step-item.completed .step-number::after {
        content: '✓';
        position: absolute;
        color: white;
        font-size: 1.5rem;
        font-weight: bold;
    }

    .step-info {
        padding-top: 5px;
    }

    .step-title {
        font-size: 1.2rem;
        font-weight: 600;
        margin-bottom: 5px;
    }

    .step-desc {
        font-size: 0.9rem;
        opacity: 0.8;
        line-height: 1.5;
    }

    /* Линия соединения шагов */
    .step-connector {
        position: absolute;
        left: 25px;
        top: 50px;
        bottom: -40px;
        width: 2px;
        background: rgba(255, 255, 255, 0.2);
        z-index: 0;
    }

    .step-item:last-child .step-connector {
        display: none;
    }

    /* Преимущества */
    .steps-benefits {
        position: relative;
        z-index: 1;
        margin-top: auto;
        padding-top: 40px;
        border-top: 1px solid rgba(255, 255, 255, 0.1);
    }

    .benefit-item {
        display: flex;
        align-items: center;
        margin-bottom: 20px;
        gap: 15px;
    }

    .benefit-icon {
        width: 40px;
        height: 40px;
        background: rgba(255, 255, 255, 0.1);
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 1.1rem;
        flex-shrink: 0;
    }

    .benefit-text {
        font-size: 0.95rem;
        opacity: 0.9;
    }

    /* Правая панель с формой */
    .register-form-section {
        flex: 1;
        padding: 60px 80px;
        display: flex;
        flex-direction: column;
        overflow-y: auto;
        max-height: 850px;
    }

    .form-progress {
        margin-bottom: 40px;
    }

    .progress-bar-container {
        height: 8px;
        background: #e0e6eb;
        border-radius: 4px;
        overflow: hidden;
        margin-bottom: 15px;
    }

    .progress-bar-fill {
        height: 100%;
        background: linear-gradient(45deg, var(--mint), var(--coral));
        border-radius: 4px;
        width: 25%;
        transition: width 0.5s ease;
    }

    .progress-text {
        display: flex;
        justify-content: space-between;
        color: var(--gray);
        font-size: 0.9rem;
    }

    .form-header {
        margin-bottom: 40px;
    }

    .form-title {
        font-size: 2.5rem;
        color: var(--dark-green);
        font-weight: 700;
        margin-bottom: 10px;
    }

    .form-subtitle {
        color: var(--gray);
        font-size: 1.1rem;
        line-height: 1.6;
    }

    /* Группы полей формы */
    .form-step {
        display: none;
        animation: fadeIn 0.5s ease;
    }

    .form-step.active {
        display: block;
    }

    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(20px); }
        to { opacity: 1; transform: translateY(0); }
    }

    .field-group {
        margin-bottom: 35px;
    }

    .field-group-title {
        font-size: 1.3rem;
        color: var(--dark-green);
        font-weight: 600;
        margin-bottom: 25px;
        padding-bottom: 15px;
        border-bottom: 2px solid rgba(78, 205, 196, 0.2);
        position: relative;
    }

    .field-group-title::after {
        content: '';
        position: absolute;
        bottom: -2px;
        left: 0;
        width: 60px;
        height: 2px;
        background: var(--mint);
    }

    .fields-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 25px;
        margin-bottom: 20px;
    }

    /* Кастомные поля формы */
    .custom-form .form-group {
        margin-bottom: 25px;
    }

    .custom-form label {
        display: block;
        margin-bottom: 10px;
        color: var(--dark-green);
        font-weight: 600;
        font-size: 0.95rem;
        letter-spacing: 0.3px;
    }

    .custom-form .form-control {
        width: 100%;
        padding: 18px 25px;
        border: 2px solid #e0e6eb;
        border-radius: 15px;
        font-size: 1rem;
        transition: all 0.3s ease;
        background: white;
        color: var(--dark);
    }

    .custom-form .form-control:focus {
        border-color: var(--mint);
        box-shadow: 0 5px 15px rgba(78, 205, 196, 0.15);
        outline: none;
    }

    .custom-form .form-control::placeholder {
        color: #a8b5c0;
    }

    .input-with-icon {
        position: relative;
    }

    .input-icon {
        position: absolute;
        left: 20px;
        top: 50%;
        transform: translateY(-50%);
        color: var(--gray);
        font-size: 1.1rem;
        z-index: 2;
    }

    .input-with-icon .form-control {
        padding-left: 55px;
    }

    /* Чекбокс согласия */
    .checkbox-group {
        margin: 40px 0;
        padding: 25px;
        background: rgba(78, 205, 196, 0.05);
        border-radius: 15px;
        border: 2px solid rgba(78, 205, 196, 0.2);
    }

    .custom-checkbox {
        display: flex;
        align-items: flex-start;
        gap: 15px;
        cursor: pointer;
    }

    .checkbox-box {
        width: 24px;
        height: 24px;
        border: 2px solid #e0e6eb;
        border-radius: 6px;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-top: 3px;
        flex-shrink: 0;
        transition: all 0.3s ease;
        background: white;
    }

    .custom-checkbox input:checked + .checkbox-box {
        background: var(--mint);
        border-color: var(--mint);
    }

    .custom-checkbox input:checked + .checkbox-box::after {
        content: '✓';
        color: white;
        font-size: 0.9rem;
        font-weight: bold;
    }

    .checkbox-text {
        flex: 1;
        color: var(--dark);
        line-height: 1.6;
    }

    .checkbox-link {
        color: var(--mint);
        text-decoration: none;
        font-weight: 500;
        transition: color 0.3s ease;
    }

    .checkbox-link:hover {
        color: var(--dark-green);
        text-decoration: underline;
    }

    /* Кнопки навигации */
    .form-navigation {
        display: flex;
        justify-content: space-between;
        margin-top: 50px;
        padding-top: 30px;
        border-top: 1px solid #e0e6eb;
    }

    .nav-button {
        padding: 16px 35px;
        border: none;
        border-radius: 15px;
        font-size: 1rem;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s cubic-bezier(0.175, 0.885, 0.32, 1.275);
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .button-prev {
        background: transparent;
        color: var(--gray);
        border: 2px solid #e0e6eb;
    }

    .button-prev:hover {
        background: #f8f9fa;
        border-color: var(--mint);
        color: var(--dark-green);
        transform: translateX(-5px);
    }

    .button-next {
        background: linear-gradient(45deg, var(--mint), #6ce6dd);
        color: white;
        margin-left: auto;
    }

    .button-next:hover {
        background: linear-gradient(45deg, #3dbdb4, #5cd4cb);
        transform: translateX(5px);
        box-shadow: 0 10px 25px rgba(78, 205, 196, 0.3);
    }

    .button-submit {
        background: linear-gradient(45deg, var(--coral), #ff8e8e);
        color: white;
        padding: 18px 50px;
    }

    .button-submit:hover {
        background: linear-gradient(45deg, #ff5a5a, #ff7c7c);
        transform: translateY(-3px);
        box-shadow: 0 15px 30px rgba(255, 107, 107, 0.3);
    }

    /* Ссылка на авторизацию */
    .login-link {
        text-align: center;
        margin-top: 40px;
        padding-top: 30px;
        border-top: 1px solid #e0e6eb;
    }

    .login-link p {
        color: var(--gray);
        margin-bottom: 15px;
        font-size: 1rem;
    }

    .login-button {
        display: inline-block;
        padding: 15px 40px;
        border: 2px solid var(--mint);
        border-radius: 50px;
        color: var(--mint);
        text-decoration: none;
        font-weight: 600;
        font-size: 1rem;
        transition: all 0.3s ease;
    }

    .login-button:hover {
        background: var(--mint);
        color: white;
        transform: translateY(-2px);
        box-shadow: 0 10px 20px rgba(78, 205, 196, 0.2);
    }

    /* Анимации */
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }

    /* Валидационные стили */
    .is-invalid .form-control {
        border-color: var(--coral) !important;
    }

    .invalid-feedback {
        color: var(--coral);
        font-size: 0.9rem;
        margin-top: 8px;
        display: block;
    }

    /* Адаптивность */
    @media (max-width: 1200px) {
        .register-container {
            flex-direction: column;
            max-width: 800px;
            min-height: auto;
        }

        .register-steps {
            flex: none;
            padding: 40px 30px;
        }

        .register-form-section {
            padding: 50px 40px;
            max-height: none;
        }

        .steps-list {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
            margin-bottom: 40px;
        }

        .step-item {
            flex: 1;
            min-width: 200px;
            margin-bottom: 0;
        }

        .step-connector {
            display: none;
        }
    }

    @media (max-width: 768px) {
        .register-form-section {
            padding: 40px 25px;
        }

        .fields-row {
            grid-template-columns: 1fr;
        }

        .form-title {
            font-size: 2rem;
        }

        .form-navigation {
            flex-direction: column;
            gap: 15px;
        }

        .nav-button {
            width: 100%;
            justify-content: center;
        }

        .steps-title {
            font-size: 2rem;
        }

        .step-item {
            min-width: 100%;
        }
    }

    @media (max-width: 576px) {
        .register-steps {
            padding: 30px 20px;
        }

        .register-form-section {
            padding: 30px 20px;
        }

        .form-title {
            font-size: 1.8rem;
        }

        .field-group-title {
            font-size: 1.1rem;
        }

        .checkbox-group {
            padding: 20px;
        }
    }
</style>

<div class="register-page">
    <div class="register-container">
        <!-- Левая панель с шагами -->
        <div class="register-steps">
            <div class="steps-header">
                <h1 class="steps-title">Начните путешествие</h1>
                <p class="steps-subtitle">Присоединяйтесь к сообществу путешественников и откройте мир новых возможностей</p>
            </div>

            <div class="steps-list">
                <div class="step-item active" id="step1">
                    <div class="step-number">1</div>
                    <div class="step-info">
                        <h3 class="step-title">Личная информация</h3>
                        <p class="step-desc">Укажите ваши основные данные для создания профиля</p>
                    </div>
                    <div class="step-connector"></div>
                </div>

                <div class="step-item" id="step2">
                    <div class="step-number">2</div>
                    <div class="step-info">
                        <h3 class="step-title">Контактные данные</h3>
                        <p class="step-desc">Добавьте email и телефон для связи</p>
                    </div>
                    <div class="step-connector"></div>
                </div>

                <div class="step-item" id="step3">
                    <div class="step-number">3</div>
                    <div class="step-info">
                        <h3 class="step-title">Документы</h3>
                        <p class="step-desc">Введите паспортные данные для бронирования</p>
                    </div>
                    <div class="step-connector"></div>
                </div>

                <div class="step-item" id="step4">
                    <div class="step-number">4</div>
                    <div class="step-info">
                        <h3 class="step-title">Безопасность</h3>
                        <p class="step-desc">Создайте надежный пароль для защиты аккаунта</p>
                    </div>
                </div>
            </div>

            <div class="steps-benefits">
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-shield-alt"></i>
                    </div>
                    <div class="benefit-text">Гарантия безопасности ваших данных</div>
                </div>
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-gift"></i>
                    </div>
                    <div class="benefit-text">Подарок при первой бронировании</div>
                </div>
                <div class="benefit-item">
                    <div class="benefit-icon">
                        <i class="fas fa-star"></i>
                    </div>
                    <div class="benefit-text">Доступ к эксклюзивным предложениям</div>
                </div>
            </div>
        </div>

        <!-- Правая панель с формой -->
        <div class="register-form-section">
            <div class="form-progress">
                <div class="progress-bar-container">
                    <div class="progress-bar-fill" id="progressBar"></div>
                </div>
                <div class="progress-text">
                    <span>Шаг <span id="currentStep">1</span> из 4</span>
                    <span id="progressPercent">25%</span>
                </div>
            </div>

            <div class="form-header">
                <h2 class="form-title">Создание аккаунта</h2>
                <p class="form-subtitle">Заполните форму ниже, чтобы присоединиться к нашему сообществу путешественников</p>
            </div>

            <?php $form = ActiveForm::begin([
                'id' => 'register-form',
                'options' => ['class' => 'custom-form'],
                'fieldConfig' => [
                    'template' => "{label}\n{input}\n{error}",
                    'errorOptions' => ['class' => 'invalid-feedback'],
                ],
            ]); ?>

            <!-- Шаг 1: Личная информация -->
            <div class="form-step active" id="formStep1">
                <div class="field-group">
                    <h3 class="field-group-title">Основные данные</h3>
                    <div class="fields-row">
                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-user input-icon"></i>
                                <?= $form->field($model, 'first_name', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => 'Иван'
                                    ]
                                ])->label('Имя')->textInput(['autofocus' => true]) ?>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-user input-icon"></i>
                                <?= $form->field($model, 'last_name', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => 'Иванов'
                                    ]
                                ])->label('Фамилия')->textInput() ?>
                            </div>
                        </div>
                    </div>

                    <div class="fields-row">
                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-user input-icon"></i>
                                <?= $form->field($model, 'patronymic', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => 'Иванович'
                                    ]
                                ])->label('Отчество')->textInput() ?>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-map-marker-alt input-icon"></i>
                                <?= $form->field($model, 'address', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => 'г. Москва, ул. Примерная, д. 1'
                                    ]
                                ])->label('Адрес')->textInput() ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Шаг 2: Контактные данные -->
            <div class="form-step" id="formStep2">
                <div class="field-group">
                    <h3 class="field-group-title">Контактная информация</h3>
                    <div class="fields-row">
                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-envelope input-icon"></i>
                                <?= $form->field($model, 'email', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => 'your.email@example.com'
                                    ]
                                ])->label('Электронная почта')->textInput() ?>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-phone input-icon"></i>
                                <?= $form->field($model, 'phone', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => '+7(999)999-99-99'
                                    ]
                                ])->widget(MaskedInput::class, [
                                    'mask' => '+7(999)999-99-99',
                                ])->label('Телефон') ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Шаг 3: Документы -->
            <div class="form-step" id="formStep3">
                <div class="field-group">
                    <h3 class="field-group-title">Документы для бронирования</h3>
                    <div class="fields-row">
                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-passport input-icon"></i>
                                <?= $form->field($model, 'passport', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => '1234 567890'
                                    ]
                                ])->widget(MaskedInput::class, [
                                    'mask' => '9999 999999',
                                ])->label('Паспорт (серия и номер)') ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Шаг 4: Безопасность -->
            <div class="form-step" id="formStep4">
                <div class="field-group">
                    <h3 class="field-group-title">Настройки безопасности</h3>
                    <div class="fields-row">
                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-lock input-icon"></i>
                                <?= $form->field($model, 'password', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => '••••••••'
                                    ]
                                ])->label('Пароль')->passwordInput() ?>
                            </div>
                        </div>

                        <div class="form-group">
                            <div class="input-with-icon">
                                <i class="fas fa-lock input-icon"></i>
                                <?= $form->field($model, 'password_repeat', [
                                    'options' => ['tag' => false],
                                    'inputOptions' => [
                                        'class' => 'form-control',
                                        'placeholder' => '••••••••'
                                    ]
                                ])->label('Повторите пароль')->passwordInput() ?>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <div class="checkbox-group">
                            <label class="custom-checkbox">
                                <?= $form->field($model, 'rules', [
                                    'options' => ['tag' => false],
                                    'template' => "{input}"
                                ])->checkbox([
                                    'class' => 'd-none'
                                ]) ?>
                                <div class="checkbox-box"></div>
                                <div class="checkbox-text">
                                    Я согласен с <a href="#" class="checkbox-link">правилами обработки персональных данных</a>
                                    и <a href="#" class="checkbox-link">условиями пользовательского соглашения</a>.
                                    Подтверждаю достоверность указанных данных.
                                </div>
                            </label>
                        </div>
                    </div>
                </div>
            </div>

            <?php ActiveForm::end(); ?>

            <!-- Навигация по шагам -->
            <div class="form-navigation">
                <button type="button" class="nav-button button-prev" id="prevButton" style="display: none;">
                    <i class="fas fa-arrow-left"></i>
                    Назад
                </button>

                <button type="button" class="nav-button button-next" id="nextButton">
                    Продолжить
                    <i class="fas fa-arrow-right"></i>
                </button>

                <button type="submit" class="nav-button button-submit" id="submitButton" style="display: none;" form="register-form">
                    <i class="fas fa-check"></i>
                    Зарегистрироваться
                </button>
            </div>

            <div class="login-link">
                <p>Уже есть аккаунт?</p>
                <a href="<?= \yii\helpers\Url::to(['site/login']) ?>" class="login-button">
                    Войти в систему
                </a>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const steps = [1, 2, 3, 4];
        let currentStep = 1;

        const progressBar = document.getElementById('progressBar');
        const currentStepSpan = document.getElementById('currentStep');
        const progressPercent = document.getElementById('progressPercent');

        const stepElements = steps.map(step => document.getElementById(`step${step}`));
        const formSteps = steps.map(step => document.getElementById(`formStep${step}`));

        const prevButton = document.getElementById('prevButton');
        const nextButton = document.getElementById('nextButton');
        const submitButton = document.getElementById('submitButton');

        // Инициализация формы
        function initForm() {
            updateStep(currentStep);
            updateProgress();
            setupFieldValidation();
            setupPasswordStrength();
        }

        // Обновление отображения шага
        function updateStep(step) {
            // Обновление шагов слева
            stepElements.forEach((el, index) => {
                const stepNumber = index + 1;
                el.classList.remove('active', 'completed');

                if (stepNumber < step) {
                    el.classList.add('completed');
                } else if (stepNumber === step) {
                    el.classList.add('active');
                }
            });

            // Обновление формы справа
            formSteps.forEach((formStep, index) => {
                const stepNumber = index + 1;
                formStep.classList.remove('active');

                if (stepNumber === step) {
                    formStep.classList.add('active');
                    // Прокрутка к началу шага
                    formStep.scrollIntoView({ behavior: 'smooth', block: 'start' });
                }
            });

            // Обновление кнопок навигации
            prevButton.style.display = step > 1 ? 'flex' : 'none';
            nextButton.style.display = step < 4 ? 'flex' : 'none';
            submitButton.style.display = step === 4 ? 'flex' : 'none';
        }

        // Обновление прогресс-бара
        function updateProgress() {
            const progress = (currentStep / 4) * 100;
            progressBar.style.width = `${progress}%`;
            currentStepSpan.textContent = currentStep;
            progressPercent.textContent = `${Math.round(progress)}%`;
        }

        // Навигация вперед
        function goToNextStep() {
            if (currentStep < 4 && validateStep(currentStep)) {
                currentStep++;
                updateStep(currentStep);
                updateProgress();
            }
        }

        // Навигация назад
        function goToPrevStep() {
            if (currentStep > 1) {
                currentStep--;
                updateStep(currentStep);
                updateProgress();
            }
        }

        // Валидация шага
        function validateStep(step) {
            let isValid = true;

            switch(step) {
                case 1:
                    const firstName = document.querySelector('input[name="RegisterForm[first_name]"]');
                    const lastName = document.querySelector('input[name="RegisterForm[last_name]"]');

                    if (!firstName.value.trim()) {
                        showError(firstName, 'Введите имя');
                        isValid = false;
                    }

                    if (!lastName.value.trim()) {
                        showError(lastName, 'Введите фамилию');
                        isValid = false;
                    }
                    break;

                case 2:
                    const email = document.querySelector('input[name="RegisterForm[email]"]');
                    const phone = document.querySelector('input[name="RegisterForm[phone]"]');

                    if (!validateEmail(email.value)) {
                        showError(email, 'Введите корректный email');
                        isValid = false;
                    }

                    if (!phone.value || phone.value.includes('_')) {
                        showError(phone, 'Введите полный номер телефона');
                        isValid = false;
                    }
                    break;

                case 3:
                    const passport = document.querySelector('input[name="RegisterForm[passport]"]');

                    if (!passport.value || passport.value.includes('_')) {
                        showError(passport, 'Введите полные паспортные данные');
                        isValid = false;
                    }
                    break;

                case 4:
                    const password = document.querySelector('input[name="RegisterForm[password]"]');
                    const passwordRepeat = document.querySelector('input[name="RegisterForm[password_repeat]"]');
                    const rules = document.querySelector('input[name="RegisterForm[rules]"]');

                    if (password.value.length < 6) {
                        showError(password, 'Пароль должен содержать не менее 6 символов');
                        isValid = false;
                    }

                    if (password.value !== passwordRepeat.value) {
                        showError(passwordRepeat, 'Пароли не совпадают');
                        isValid = false;
                    }

                    if (!rules.checked) {
                        alert('Пожалуйста, примите условия пользовательского соглашения');
                        isValid = false;
                    }
                    break;
            }

            return isValid;
        }

        // Валидация email
        function validateEmail(email) {
            const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return re.test(email);
        }

        // Показать ошибку
        function showError(input, message) {
            const formGroup = input.closest('.form-group');
            let errorDiv = formGroup.querySelector('.invalid-feedback');

            if (!errorDiv) {
                errorDiv = document.createElement('div');
                errorDiv.className = 'invalid-feedback';
                formGroup.appendChild(errorDiv);
            }

            errorDiv.textContent = message;
            input.classList.add('is-invalid');
            errorDiv.style.display = 'block';

            // Анимация ошибки
            input.style.animation = 'none';
            setTimeout(() => {
                input.style.animation = 'pulse 0.5s ease';
            }, 10);
        }

        // Очистить ошибку
        function clearError(input) {
            const formGroup = input.closest('.form-group');
            const errorDiv = formGroup.querySelector('.invalid-feedback');

            if (errorDiv) {
                errorDiv.style.display = 'none';
            }

            input.classList.remove('is-invalid');
        }

        // Настройка валидации полей
        function setupFieldValidation() {
            const inputs = document.querySelectorAll('.custom-form .form-control');

            inputs.forEach(input => {
                input.addEventListener('focus', function() {
                    clearError(this);
                });

                input.addEventListener('blur', function() {
                    // Базовая валидация при потере фокуса
                    if (!this.value.trim() && this.hasAttribute('required')) {
                        showError(this, 'Это поле обязательно для заполнения');
                    }
                });
            });
        }

        // Индикатор сложности пароля
        function setupPasswordStrength() {
            const passwordInput = document.querySelector('input[name="RegisterForm[password]"]');
            const passwordRepeatInput = document.querySelector('input[name="RegisterForm[password_repeat]"]');

            if (passwordInput) {
                passwordInput.addEventListener('input', function() {
                    const password = this.value;
                    let strength = 0;

                    if (password.length >= 6) strength++;
                    if (password.length >= 8) strength++;
                    if (/[A-Z]/.test(password)) strength++;
                    if (/[0-9]/.test(password)) strength++;
                    if (/[^A-Za-z0-9]/.test(password)) strength++;

                    // Визуализация сложности пароля
                    const indicator = this.parentNode.querySelector('.password-strength') ||
                        (function() {
                            const div = document.createElement('div');
                            div.className = 'password-strength';
                            div.style.marginTop = '8px';
                            div.style.height = '4px';
                            div.style.borderRadius = '2px';
                            div.style.background = '#e0e6eb';
                            div.style.overflow = 'hidden';
                            this.parentNode.appendChild(div);
                            return div;
                        }).call(this);

                    const strengthBar = indicator.querySelector('.strength-bar') ||
                        (function() {
                            const bar = document.createElement('div');
                            bar.className = 'strength-bar';
                            bar.style.height = '100%';
                            bar.style.width = '0%';
                            bar.style.transition = 'width 0.3s ease';
                            indicator.appendChild(bar);
                            return bar;
                        }).call(this);

                    const width = (strength / 5) * 100;
                    strengthBar.style.width = `${width}%`;

                    if (strength <= 2) {
                        strengthBar.style.background = 'var(--coral)';
                    } else if (strength <= 4) {
                        strengthBar.style.background = 'var(--yellow)';
                    } else {
                        strengthBar.style.background = 'var(--mint)';
                    }
                });
            }

            if (passwordRepeatInput) {
                passwordRepeatInput.addEventListener('input', function() {
                    const password = passwordInput.value;
                    const repeat = this.value;

                    if (repeat && password !== repeat) {
                        showError(this, 'Пароли не совпадают');
                    } else {
                        clearError(this);
                    }
                });
            }
        }

        // Обработка чекбокса согласия
        const rulesCheckbox = document.querySelector('input[name="RegisterForm[rules]"]');
        const checkboxBox = document.querySelector('.checkbox-box');

        if (checkboxBox) {
            checkboxBox.addEventListener('click', function() {
                rulesCheckbox.checked = !rulesCheckbox.checked;
                this.classList.toggle('checked', rulesCheckbox.checked);
            });
        }

        // Обработчики кнопок
        nextButton.addEventListener('click', goToNextStep);
        prevButton.addEventListener('click', goToPrevStep);

        // Отправка формы
        const registerForm = document.getElementById('register-form');
        if (registerForm) {
            registerForm.addEventListener('submit', function(e) {
                if (validateStep(4)) {
                    const submitBtn = this.querySelector('#submitButton');
                    if (submitBtn) {
                        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Регистрация...';
                        submitBtn.disabled = true;
                    }
                } else {
                    e.preventDefault();
                }
            });
        }

        // Анимация появления элементов
        setTimeout(() => {
            document.querySelectorAll('.form-step.active .form-group').forEach((el, index) => {
                el.style.opacity = '0';
                el.style.transform = 'translateY(20px)';

                setTimeout(() => {
                    el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
                    el.style.opacity = '1';
                    el.style.transform = 'translateY(0)';
                }, 100 + index * 100);
            });
        }, 300);

        // Инициализация
        initForm();
    });
</script>